import glob
from pathlib import Path
import torch as t
import pandas as pd
import matplotlib.pyplot as plt
import seaborn as sns



def load_activations_and_labels(activations_dir, layer, device):
    """
    Load all activations and labels for a specific layer from the saved files.
    Returns list of activation tensors (since they have different sequence lengths) and concatenated labels tensor.
    """
    activation_files = sorted(activations_dir.glob(f"layer_{layer}_*.pt"))
    label_files = sorted(activations_dir.glob("labels_*.pt"))

    # Load activations (keep as list since they have different sequence lengths)
    activations = []
    for act_file in activation_files:
        activations.append(t.load(act_file))

    # Load and concatenate all labels
    labels = []
    for label_file in label_files:
        labels.append(t.tensor(t.load(label_file), device=device))

    labels = t.cat(labels, dim=0)

    return activations, labels.to(device)

def save_results(scores, labels, results, model_name, layer, dataset_name, probe_path=None, output_dir=None):
    # Create a list of results dictionaries, one for each token
    all_results = []
    for token, metrics in results.items():
        token_results = {
            'model': model_name,
            'layer': layer,
            'dataset': dataset_name,
            'probe_path': probe_path,
            'token': token,
            'auroc': metrics['auroc'],
            'accuracy': metrics['accuracy'],
            'balanced_accuracy': metrics['balanced_accuracy'],
            'recall': metrics['recall']
        }
        all_results.append(token_results)
    
    output_path = Path(output_dir) / dataset_name / str(layer)
    output_path.mkdir(parents=True, exist_ok=True)
    
    # Save all results to a single file
    results_file = output_path / f"evaluation_results_layer_{layer}.csv"
    results_df = pd.DataFrame(all_results)
    results_df.to_csv(results_file, index=False)
    print(f"Saved results to {results_file}")

    # Save scores and visualizations for each token
    for token, metrics in results.items():
        # Save scores for this token
        t.save(scores[token], output_path / f"scores_{token}.pt")
        
        # Visualize predictions for this token
        visualize_preds(scores[token], labels, output_path, model_name, layer, token, dataset_name)

    return all_results

def visualize_preds(scores, labels, output_dir, model_name, layer, token, dataset_name):
    """
    Visualize predictions vs labels using a scatter plot.
    
    Args:
        scores: Tensor of probe scores
        labels: Tensor of true labels
    """
    # Convert tensors to float32 before numpy conversion
    scores = scores.to(t.float32).cpu().numpy()
    mask = labels.to(t.float32).bool().cpu().numpy()

    # plot 
    plt.hist(scores[mask], bins=25, label="truths", alpha=0.5, color="blue")
    plt.hist(scores[~mask], bins=25, label="lies", alpha=0.5, color="red")
    plt.title(f"probe scores for {model_name} layer {layer} token {token}")
    plt.legend()

    # save image
    output_dir.mkdir(parents=True, exist_ok=True)
    path = Path(output_dir) / f"probe_scores_{model_name}_layer_{layer}_{token}_{dataset_name}.png".replace("/", "_")
    plt.savefig(path)

def filter_dataset(dataset, model_name):
    """
    Filter dataset to only include statements with a valid label
    """
    model_names_mapping = {
        "Qwen/Qwen2.5-72B-Instruct": "qwen-2.5-72b-instruct",
        "meta-llama/Llama-3.3-70B-Instruct": "llama-v3.3-70b-instruct", 
        "mistralai/Mistral-Small-3.1-24B-Instruct-2503": "mistral-small-3.1-24b-instruct",
    }
    model_name = model_names_mapping[model_name]
    filtered_dataset = dataset[dataset['model'].str.contains(model_name)]
    filtered_dataset = filtered_dataset[filtered_dataset['dishonest'] != '-1']

    return filtered_dataset